module net.BurtonRadons.dig.platform.canvasGL;

private import net.BurtonRadons.dig.platform.base;
private import net.BurtonRadons.dig.platform.canvas;
private import net.BurtonRadons.dig.platform.control;
private import net.BurtonRadons.dig.gl;
//private import std.c.windows.windows;

// version = _POINT_SAMPLE; // use point sampling in image scaling rather than multisampling.

//alias net.BurtonRadons.dig.gl.HANDLE  HANDLE; //Added by Joel Anderson
alias net.BurtonRadons.dig.platform.windows._HANDLE  HANDLE; //Added by Joel Anderson
alias net.BurtonRadons.dig.platform.windows._HDC  HDC;
alias net.BurtonRadons.dig.platform.windows._BOOL  BOOL;
alias net.BurtonRadons.dig.platform.windows._HBRUSH  HBRUSH;
//alias net.BurtonRadons.dig.platform.windows.toWStringz  toWStringz;
//alias net.BurtonRadons.dig.platform.windows.wtoStringz wtoStringz;


//alias HANDLE HDC;		 		//Added by Joel Anderson
alias int _BOOL;				 		//Added by Joel Anderson
alias HANDLE HGLRC;   		//Added by Joel Anderson
alias LPSTR LPTCH, PTCH;	//Added by Joel Anderson
alias char CHAR;					//Added by Joel Anderson
alias CHAR *LPSTR;   			//Added by Joel Anderson
alias CHAR *PSTR;   			//Added by Joel Anderson
alias CHAR *LPCSTR;   		//Added by Joel Anderson
alias CHAR *PCSTR;   			//Added by Joel Anderson
alias LPSTR PTSTR, LPTSTR; //Added by Joel Anderson
alias LPCSTR _LPCTSTR;   		//Added by Joel Anderson


alias void GLvoid;
alias ubyte GLboolean;
alias int GLenum;
alias uint GLbitfield;

alias byte GLbyte;
alias ubyte GLubyte;

alias short GLshort;
alias ushort GLushort;

alias int GLint;
alias int GLsizei;
alias uint GLuint;

alias float GLclampf;
alias float GLfloat;

alias double GLdouble;
alias double GLclampd;

private import net.BurtonRadons.dig.common.math;

/** This creates a frame which holds an OpenGL context.
  When using this class, you need to include an additional
  library in your commands, the "opengl32.lib" library, unless if
  you use dig (which can detect this usage).  A very, very small 
  version of it is copied automatically into your lib directory. 
  In order to reduce dependencies, the CanvasGL header is not
  imported by dig; use "import diggl;" as well.

  Color index mode is not supported either in the contexts or in
  the API.

  The GL class and gl std.math.singleton wraps some of OpenGL's functionality
  in a more convenient wrapper.  It's not slower; everything is in
  final methods that will like as not be inlined.  I make minimal
  design choices: multipurpose functions like glFog are split into
  component methods, there are no prefixes for either methods or
  names, and pointless separate functions like glDepthMask are
  coralled into enable/disable/isEnabled.  The texturing API is
  completely different and pulls everything into a "texture" prefix,
  likewise "list".

  One guiding factor of the API is a lack of buffered state; thus it
  can interoperate with any other OpenGL code, and it keeps it
  focused upon providing a wrapper, not a framework.

  <h2>The Depth Buffer</h2>

  OpenGL state includes the depth buffer, a buffer that holds the
  individual depth of each pixel on the color buffer.  CanvasGL asks
  for a depth buffer with 32-bit precision; in modern hardware this
  will be provided, but old hardware might give 16-bit.  Lower
  precision increases z-fighting, where the intersection between two
  polygons start shivering as one polygon or the other is seen as
  closer.

  Initial GL state doesn't use the depth buffer and must be enabled
  with #enable (<b>DEPTH_TEST</b>).

  To clear the depth buffer, use #clear (<b>DEPTH_BUFFER_BIT</b>).

  The value that the depth buffer is cleared to is controlled by
  #clearDepth, which is initially 1.

  Once we have the depth for a fragment we still need to scale it into
  the depth buffer so that it can be used for comparisons.  This is
  controlled with the #depthRange function.

  The depth function is used to determine whether any given fragment
  wins a contest with the previous depth fragment.  This is controlled
  by #depthFunc.  The initial function is <b>LESS</b>.

  Once the point wins the contest it may be written depending upon the
  value of #enable (DEPTH_WRITEMASK), which is true by default.

  The most important way to maximise depth buffer usage is to keep the
  ratio of the far plane and the near plane as low as possible.  The
  higher it is, the more depth buffer precision is lost.  The two planes
  are controlled by #frustum.

  <h2>Lighting</h2>

  By default, lighting is disabled, thus causing everything to be drawn
  fullbright.  When enabled with #enable (<b>LIGHTING</b>), lighting is
  multiplied to a color after texturing but before fogging.  While lighting
  is disabled, #color indicates the color of an object; after lighting, the
  material parameters are used.

  GL offers two ways to access lights.  You can use the lightXXX
  method, which takes a light index (light indices start with <b>LIGHT0</b>
  and are sequential), or use the #getLight method and use the methods
  on the returned GLlight object.  They have identical parameters outside of
  the first index.  The number of lights the GL supports can be retrieved
  using #maxLights ().

  Lights must be enabled individually using #enable (<b>LIGHTi</b>), or with
  the @a enable method on the GLlight.  No lights are initially enabled.
  
  Then you can change the diffuse color, with #lightDiffuse/@a diffuse.  This
  is the color that the light reflects off an object in all directions; a
  diffusely lit sphere has a warm glow that fades to black as it faces further
  from the light.  By contrast, the specular color (#lightSpecular/@a specular)
  is the reflection of the light off the sphere.  This assumes that the light is
  circular.  The ambient color (#lightAmbient/@a ambient) controls the light's
  brightness when it's not shining on the object.  The position
  (#lightPosition/@a position) is where the light is located.  When you set
  this value, it is first multiplied against the modelview matrix.

  Then there's attenuation, which is how the effect of the light fades over
  a distance, and is the inverse of the addition of three factors: constant
  attenuation (distance has no effect), linear attenuation (doubling the
  distance halves the brightness), and quadratic attenuation (doubling the
  distance quarters the brightness), playfully named
  #lightConstantAttenuation/@a constantAttenuation,
  #lightLinearAttenuation/@a linearAttenuation, and
  #lightQuadraticAttenuation/@a quadraticAttenuation.  By default these values
  are (1, 0, 0), resulting in no attenuation.  Real life has the factors
  (0, y, 1) where y is the atmospheric effect, but this can be hard to model
  with, so it is more common to use linear attenuation.  Note, however, that
  you are free to use (0, 0, 0.1) or lower to change the lower limit for the
  light's attenuation.  You can set all three parameters simultaneously using
  the #lightAttenuation/@a attenuation methods.

  Spotlights are another form of light that project only a certain angular
  range.  Like normal lights, OpenGL spotlights are vertex only; this means
  that if an object such as a street is not sufficiently tesselated, the
  spotlight might look rough or even be invisible.  This limitation makes it
  tricky to use OpenGL spotlights, but they are available.

  The first spotlight parameter is #lightSpotCutoff/@a spotCutoff.  When 180
  (the default), this is a normal diffuse light; otherwise it is the spread
  angle of the spot, from 0 to 90 degrees.  90 degrees results in a
  half-sphere of light, for example, while 1 degree results in a small circle.

  Then there's #lightSpotExponent/@a spotExponent, which controls the falloff.
  As a given point approaches the spotlight's edges, it gradually approaches black.
  When the exponent is 0, this approach is linear.  When the exponent is 128,
  this is a sudden sharp line at the edge of the cutoff.

  Finally, there's #lightSpotDirection/@a spotDirection, which is the normal
  of the light's direction.  It is transformed by the inverse of the modelview
  matrix.

  <h2>Material Parameters</h2>

  What effect this all has on an object depends upon the object's material
  parameters.  Each side to an object has different material parameters.  The
  front side is determined by the triangle's winding, which is whether a
  triangle's points are clockwise in order or counterclockwise.  Which
  direction is considered front-facing depends upon the #frontFace parameter.
  When <b>LIGHT_MODEL_TWO_SIDE</b> is false (the default), only the front face is
  used for lighting; if the object is facing away from the light, then it is
  considered to have no intensity.

  Material parameters mirror the light's, and each is multiplied against the
  other.  These parameters are #materialAmbient, #materialDiffuse, and
  #materialSpecular.  Materials add the #materialEmission, which is an ambient
  value that is not multiplied against anything (thus, a (1, 1, 1) emission
  always results in a white object).  They also add the #materialShininess,
  which is an exponent from 0 to 128, and determines how sharp the specular
  highlight is.  The closer to 128 it gets, the sharper its edge.  Finally,
  you can use #materialAmbientAndDiffuse to set both parameters to the same
  color.

  You're not done with my short summary.  Whether
  <b>LIGHT_MODEL_LOCAL_VIEWER</b> is enabled controls how the location of the
  specular highlight is determined.  #lightModelAmbient is the final ambient
  multiplier to determine actual ambience; that makes it the best candidate
  for use.

  How the specular highlight works is one of the deficiencies of OpenGL.  Or,
  rather, how the resultant color is blended with the texture, which is to
  multiply it against it.  That means that the specular highlight on a black
  nontextured object can be white, while the highlight on a black textured
  object will be black.  To add the specular color onto the fragment color
  after multiplying the light and texture color together,
  #enable (<b>SEPARATE_SPECULAR_COLOR</b>).

  <h2>Fogging</h2>

  Fogging interpolates a color onto the texture color.  The effect of
  this color depends upon the depth of the point and the fogging
  parameters.  Fogging is initially disabled; enable it using #enable (<b>FOG</b>).

  The equation used to determine the fog factor depends upon the setting
  in #fogMode, which is initially <b>EXP</b>.  The potential values and their
  equations (where @a e is the setting in #fogEnd, @a s is the setting in #fogStart,
  @a d is the setting in #fogDensity, and @a z is the depth of this point):

  <b>LINEAR</b>: f = (e - z) / (e - s)
  \n <b>EXP</b>: f = exp (-d * z)
  \n <b>EXP2</b>: f = exp (-(d * z) ^ 2)

  The result is then clamped to the range [0, 1]; if 1, the fog color is
  used (specified with #fogColor); if 0, the texture color is used; values
  in between are interpolated.

  By default, the depth of a point is the distance of that point from the
  camera plane.  This is not as accurate as using the distance from the camera,
  and results in objects being less fogged at the corner of the vision as they
  are at the center of it.  If the proper extensions are supported, you can
  control this with #fogDistanceMode.

  Fogging interferes with drawing by producing odd colors when the blend
  mode is not (ONE, ZERO) or (SRC_ALPHA, ONE_MINUS_SRC_ALPHA).  You should find
  some way to accomodate that or disable fogging for these entities.

  <h2>Blending</h2>

  If you've been following along sequentially, you now know we have a textured,
  lit, fogged color.  How this is put onto the screen is a matter of blending.
  When blending is disabled, we skip past the next paragraph.  If
  #isEnabled (<b>BLEND</b>), however, we keep going.

  The fragment color and the destination color (the color presently in the
  color buffer) are multiplied based on the settings in #blendFunc.  The most
  popular setting is #blendFunc (<b>SRC_ALPHA</b>, <b>ONE_MINUS_SRC_ALPHA</b>).
  These two values are then combined using the #blendEquation; it's conceivable
  for this to be unsupported, in which case the colors are always added
  together.

  We are now in the end-game.  The color is now potentially blended again with
  the destination color if #enable (<b>COLOR_LOGIC_OP</b>); you control the
  logic operator used with #logicOp.  I warn you that this is not an
  accelerated path on important hardware and may not be accelerated on any
  hardware.

  Then the channels of the frame buffer are written to.  Which channels are
  written depends upon the settings of the #colorMask.  Once that is done we
  are finished with the fragment.

  <h2>Rendering Primitives</h2>

  Primitives are rendered in a #begin and #end section of code.  See #begin for
  details on each mode; here I just cover some basics and fine details that
  don't belong anywhere else.

  Each vertex of the primitive is defined using the #vertex command.  This uses
  the current parameters, defined by: #color, #edgeFlag, #index, #normal, and
  #textureCoord.
  
  Inside of the #begin/#end block, you can run a limited number of GL commands
  only.  Specifically, these are available: #vertex, #color, #index, #normal,
  #textureCoord, #evalCoord, #evalPoint, #arrayElement, #material, #edgeFlag,
  #callList, and #callLists.  No other command is guaranteed to work and should
  generate an error, although there are a small number of commands that error
  on GL discretion.

  OpenGL does not declare how a polygon or quad is broken into triangles, and
  many implementations split them based on orientation, rather than their order
  of input.  This results in severe visual discrepancies when lighting or when
  non-textured.  Moreover, OpenGL allows polygons to be triangulated with no
  regard for concavity, so most implementations simply treat them as if they
  are convex, which results in incorrect rendering if that's not true.
  Therefore, you should not use <b>POLYGON</b>, <b>QUADS</b>, or
  <b>QUAD_STRIP</b> unless if it has a static orientation (such as for a
  screen icon), or if you're not lighting but are texturing.

  <h2>2D Texturing</h2>

  Textures are referred to by an abstract number called the name.  GL
  has a different API from OpenGL's when it comes to textures.  I call 2D
  textures textures (1D and 3D textures will have to get their own names)
  and name everything with a texture prefix, rather than ugly names like
  texXXX or arbitrary swapped names like bindTexture.  There are two APIs
  for textures, using either names or an object.  I only cover the object
  API here, as the name API is identical just more annoying.

  You create a texture using #textureNew.  See the @ref GLtexture class
  for more details.

  */

class CanvasGL : Control
{
    private import net.BurtonRadons.dig.platform.frame;
    private import net.BurtonRadons.dig.platform.windows;

    /* Setup the GUI class for windows. */
    static this ()
    {
        digPlatformWindowClass.style = CS_OWNDC;
        digPlatformWindowClass.lpfnWndProc = &digPlatformWindowProc;
        digPlatformWindowClass.cbClsExtra = 0;
        digPlatformWindowClass.cbWndExtra = 0;
        digPlatformWindowClass.hInstance = hinst;
        digPlatformWindowClass.hCursor = LoadCursorA ((HANDLE) 0, IDC_ARROW);
        digPlatformWindowClass.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
        digPlatformWindowClass.lpszMenuName = null;
        digPlatformWindowClass.lpszClassName = wtoStringz (digPlatformClassName);
        std.c.windows.windows.RegisterClassA (&digPlatformWindowClass);

        gl = new GL ();
    }

    /** This associative list contains a deconstructed form of the 
      * #getString (EXTENSIONS) command; each key is an extension,
      * each value is false.  You can use it by the form:
      * "'SGIS_fog_function' in extensions"; DON'T use
      * "extensions ['SGIS_fog_function']" (values are cleared to
      * false to make this invalid).
      */

    int [char []] extensions;
    GLint textureUnitCount; /**< The number of texture units available. */

    /** Register with the parent and create the widget. */
    this (Control parent)
    {
        super (parent);
        digPlatformStyle = WS_VISIBLE | WS_CHILD | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;// WS_OVERLAPPEDWINDOW;
        digPlatformHWNDCreate (0, digPlatformClassName, null, digPlatformStyle, (HANDLE) 0);
        digPlatformCreateContext ();
    }

    /** Get a function pointer return it. */
    void *findProcedure (char [] name)
    {
        return wglGetProcAddress (std.string.toStringz (name));
    }

    /** Delete the OS-specific data */
    ~this ()
    {
		wglMakeCurrent ((HANDLE) 0, (HANDLE) 0);
        wglDeleteContext (digPlatformGLRC);
    }

    /** Make this the current OpenGL context. */
    void makeCurrent ()
    {
        if (!wglMakeCurrent (digPlatformHDC, digPlatformGLRC))
            digPlatformGetLastError ("CanvasGL.beginPaint (wglMakeCurrent)");
        gl.digCommonCurrentCanvas = this;
    }

    /** Make this the current OpenGL context and prepare for painting. */
    void beginPaint ()
    {
        std.c.windows.windows.BeginPaint (digPlatformHWND, &digPlatformPainter);
        makeCurrent ();
        gl.viewport (0, 0, width (), height ());
    }

    /** Finish painting; display the current buffer. */
    void endPaint ()
    {
        glFlush ();
        SwapBuffers (digPlatformHDC);
        ReleaseDC (digPlatformHWND, digPlatformHDC);
        //wglMakeCurrent (0, 0);
        std.c.windows.windows.EndPaint (digPlatformHWND, &digPlatformPainter);
    }

    /** Assign whether to use fullscreen mode.  When fullscreen, this canvas
      * envelops the entire screen and the mode is set to the suggested width
      * and height.  Resetting to windowed mode returns it to normal.
      */
    void fullscreen (bit value)
    {
        if (value == digPlatformFullscreen)
            return;
        digPlatformFullscreen = value;

        if (value)
        {
            DEVMODEA mode;
            mode.dmSize = mode.size;
            mode.dmPelsWidth = digPlatformSuggestWidth;
            mode.dmPelsHeight = digPlatformSuggestHeight;
            mode.dmFields = DM_PELSWIDTH | DM_PELSHEIGHT;
            ChangeDisplaySettingsA (&mode, CDS_FULLSCREEN);
        }

        digPlatformRecreate ();
    }
    
/+
#ifdef DoxygenMustSkipThis
+/

    const wchar [] digPlatformClassName = "digPlatformCanvasGL";
    
    static _WNDCLASS digPlatformWindowClass;

    _PAINTSTRUCT digPlatformPainter;
    _HANDLE digPlatformGLRC;
    bit digPlatformFullscreen = false;

    void digPlatformCreateContext ()
    {
        std.c.windows.windows.PIXELFORMATDESCRIPTOR info;
        int index;

        /* Construct the pixel format */
        info.nSize = info.size;
        info.nVersion = 1;
        info.dwFlags = PFD_DRAW_TO_WINDOW
                     | PFD_SUPPORT_OPENGL
                     | PFD_DOUBLEBUFFER;
        info.iPixelType = PFD_TYPE_RGBA;
        info.cColorBits = 24;
        info.cAlphaBits = 8;

        info.cAccumBits = 0;
        info.cDepthBits = 32;
        info.cStencilBits = 8;
        info.cAuxBuffers = 0;
        info.iLayerType = PFD_MAIN_PLANE;

        if ((index = ChoosePixelFormat (digPlatformHDC, &info)) == 0)
            digPlatformGetLastError ("CanvasGL.this (ChoosePixelFormat)");

        if (!SetPixelFormat (digPlatformHDC, index, &info))
            digPlatformGetLastError ("CanvasGL.this (SetPixelFormat)");

        digPlatformGLRC = wglCreateContext (digPlatformHDC);
        if (digPlatformGLRC == (_HANDLE) 0)
            digPlatformGetLastError ("CanvasGL.this (wglCreateContext)");

        if (!wglMakeCurrent (digPlatformHDC, digPlatformGLRC))
            digPlatformGetLastError ("CanvasGL.beginPaint (wglMakeCurrent)");

        char [] exts;
        char [] [] array;

        extensions = null;

        exts = gl.getString (GL.EXTENSIONS);
        array = std.string.split (exts);

        for (int c; c < array.length; c ++)
            extensions [array [c]] = false;

        delete array;
        
        *(void **) &glBlendEquationEXT = findProcedure ("glBlendEquationEXT");
        *(void **) &glActiveTexture = findProcedure ("glActiveTexture");
        *(void **) &glClientActiveTexture = findProcedure ("glClientActiveTexture");
        *(void **) &glMultiTexCoord1d = findProcedure ("glMultiTexCoord1d");
        *(void **) &glMultiTexCoord1dv = findProcedure ("glMultiTexCoord1dv");
        *(void **) &glMultiTexCoord1f = findProcedure ("glMultiTexCoord1f");
        *(void **) &glMultiTexCoord1fv = findProcedure ("glMultiTexCoord1fv");
        *(void **) &glMultiTexCoord1i = findProcedure ("glMultiTexCoord1i");
        *(void **) &glMultiTexCoord1iv = findProcedure ("glMultiTexCoord1iv");
        *(void **) &glMultiTexCoord1s = findProcedure ("glMultiTexCoord1s");
        *(void **) &glMultiTexCoord1sv = findProcedure ("glMultiTexCoord1sv");
        *(void **) &glMultiTexCoord2d = findProcedure ("glMultiTexCoord2d");
        *(void **) &glMultiTexCoord2dv = findProcedure ("glMultiTexCoord2dv");
        *(void **) &glMultiTexCoord2f = findProcedure ("glMultiTexCoord2f");
        *(void **) &glMultiTexCoord2fv = findProcedure ("glMultiTexCoord2fv");
        *(void **) &glMultiTexCoord2i = findProcedure ("glMultiTexCoord2i");
        *(void **) &glMultiTexCoord2iv = findProcedure ("glMultiTexCoord2iv");
        *(void **) &glMultiTexCoord2s = findProcedure ("glMultiTexCoord2s");
        *(void **) &glMultiTexCoord2sv = findProcedure ("glMultiTexCoord2sv");
        *(void **) &glMultiTexCoord3d = findProcedure ("glMultiTexCoord3d");
        *(void **) &glMultiTexCoord3dv = findProcedure ("glMultiTexCoord3dv");
        *(void **) &glMultiTexCoord3f = findProcedure ("glMultiTexCoord3f");
        *(void **) &glMultiTexCoord3fv = findProcedure ("glMultiTexCoord3fv");
        *(void **) &glMultiTexCoord3i = findProcedure ("glMultiTexCoord3i");
        *(void **) &glMultiTexCoord3iv = findProcedure ("glMultiTexCoord3iv");
        *(void **) &glMultiTexCoord3s = findProcedure ("glMultiTexCoord3s");
        *(void **) &glMultiTexCoord3sv = findProcedure ("glMultiTexCoord3sv");
        *(void **) &glMultiTexCoord4d = findProcedure ("glMultiTexCoord4d");
        *(void **) &glMultiTexCoord4dv = findProcedure ("glMultiTexCoord4dv");
        *(void **) &glMultiTexCoord4f = findProcedure ("glMultiTexCoord4f");
        *(void **) &glMultiTexCoord4fv = findProcedure ("glMultiTexCoord4fv");
        *(void **) &glMultiTexCoord4i = findProcedure ("glMultiTexCoord4i");
        *(void **) &glMultiTexCoord4iv = findProcedure ("glMultiTexCoord4iv");
        *(void **) &glMultiTexCoord4s = findProcedure ("glMultiTexCoord4s");
        *(void **) &glMultiTexCoord4sv = findProcedure ("glMultiTexCoord4sv");

        glPixelStorei (GL.PACK_ALIGNMENT, 1);
        glPixelStorei (GL.UNPACK_ALIGNMENT, 1);
        makeCurrent ();

        glGetIntegerv (GL.MAX_TEXTURE_UNITS, &textureUnitCount);
    }
    
    static extern (Windows)
    _LRESULT digPlatformWindowProc (_HWND hwnd, _UINT message, _WPARAM wParam, _LPARAM lParam)
    {
        Control control = digPlatformHWNDToControl [hwnd];
        int result = 0;
        Event event;

        switch (message)
        {
            case WM_PAINT:
                control.onPaint.notify ();
                break;

            case WM_ERASEBKGND:
                return true;

            default:
                return Frame.digPlatformWindowProc (hwnd, message, wParam, lParam);
        }

        return result;
    }

    extern (Windows)
    {
        void (*glBlendEquationEXT) (GLenum mode);

        void (*glActiveTexture) (GLenum texture);
        void (*glClientActiveTexture) (GLenum texture);
        void (*glMultiTexCoord1d) (GLenum target, GLdouble s);
        void (*glMultiTexCoord1dv) (GLenum target, GLdouble *v);
        void (*glMultiTexCoord1f) (GLenum target, GLfloat s);
        void (*glMultiTexCoord1fv) (GLenum target, GLfloat *v);
        void (*glMultiTexCoord1i) (GLenum target, GLint s);
        void (*glMultiTexCoord1iv) (GLenum target, GLint *v);
        void (*glMultiTexCoord1s) (GLenum target, GLshort s);
        void (*glMultiTexCoord1sv) (GLenum target, GLshort *v);
        void (*glMultiTexCoord2d) (GLenum target, GLdouble s, GLdouble t);
        void (*glMultiTexCoord2dv) (GLenum target, GLdouble *v);
        void (*glMultiTexCoord2f) (GLenum target, GLfloat s, GLfloat t);
        void (*glMultiTexCoord2fv) (GLenum target, GLfloat *v);
        void (*glMultiTexCoord2i) (GLenum target, GLint s, GLint t);
        void (*glMultiTexCoord2iv) (GLenum target, GLint *v);
        void (*glMultiTexCoord2s) (GLenum target, GLshort s, GLshort t);
        void (*glMultiTexCoord2sv) (GLenum target, GLshort *v);
        void (*glMultiTexCoord3d) (GLenum target, GLdouble s, GLdouble t, GLdouble r);
        void (*glMultiTexCoord3dv) (GLenum target, GLdouble *v);
        void (*glMultiTexCoord3f) (GLenum target, GLfloat s, GLfloat t, GLfloat r);
        void (*glMultiTexCoord3fv) (GLenum target, GLfloat *v);
        void (*glMultiTexCoord3i) (GLenum target, GLint s, GLint t, GLint r);
        void (*glMultiTexCoord3iv) (GLenum target, GLint *v);
        void (*glMultiTexCoord3s) (GLenum target, GLshort s, GLshort t, GLshort r);
        void (*glMultiTexCoord3sv) (GLenum target, GLshort *v);
        void (*glMultiTexCoord4d) (GLenum target, GLdouble s, GLdouble t, GLdouble r, GLdouble q);
        void (*glMultiTexCoord4dv) (GLenum target, GLdouble *v);
        void (*glMultiTexCoord4f) (GLenum target, GLfloat s, GLfloat t, GLfloat r, GLfloat q);
        void (*glMultiTexCoord4fv) (GLenum target, GLfloat *v);
        void (*glMultiTexCoord4i) (GLenum target, GLint s, GLint t, GLint r, GLint q);
        void (*glMultiTexCoord4iv) (GLenum target, GLint *v);
        void (*glMultiTexCoord4s) (GLenum target, GLshort s, GLshort t, GLshort r, GLshort q);
        void (*glMultiTexCoord4sv) (GLenum target, GLshort *v);
    }
    
    override void digPlatformRecreate () //Modified by Joel Anderson to do a fullscreen like it should be.
    {
        digPlatformStyle = WS_VISIBLE | WS_CHILD | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;
        if (digPlatformFullscreen)
            digPlatformStyle = WS_VISIBLE | WS_MAXIMIZE | WS_POPUP;

        wglMakeCurrent ((HANDLE) 0, (HANDLE) 0);
        wglDeleteContext (digPlatformGLRC);
        DestroyWindow (digPlatformHWND);
        delete digPlatformHWNDToControl [digPlatformHWND];
        //hwndCreate (0, className, setCaption, style, null);
        HANDLE parent;
        int vleft = std.c.windows.windows.CW_USEDEFAULT;
        int vtop = std.c.windows.windows.CW_USEDEFAULT;
        int vwidth = width ();
        int vheight = height ();

        if (this.parent () !== null)
            parent = this.parent ().digPlatformHWND;
        if (digPlatformFullscreen)
        {
            parent = (HANDLE) 0;
            vleft = 0;
            vtop = 0;
        }

        digPlatformHWNDCreate(WS_EX_APPWINDOW | (digPlatformFullscreen)?WS_EX_TOPMOST:0, digPlatformClassName, digPlatformCaption, digPlatformStyle, vleft, vtop, vwidth, vheight, parent, (HANDLE) 0, hinst, null);
        digPlatformCreateContext ();
        digPlatformMoved ();
        childIterate (delegate void (Control child) { child.digPlatformRecreate (); });
        makeCurrent ();
    }

   /+ override void digPlatformRecreate ()
    {
        digPlatformStyle = WS_VISIBLE | WS_CHILD | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;// WS_OVERLAPPEDWINDOW;
        if (digPlatformFullscreen)
            digPlatformStyle = WS_VISIBLE | WS_OVERLAPPEDWINDOW;

        wglMakeCurrent ((_HANDLE) 0, (_HANDLE) 0);
        wglDeleteContext (digPlatformGLRC);
        DestroyWindow (digPlatformHWND);
        delete digPlatformHWNDToControl [digPlatformHWND];
        //hwndCreate (0, className, setCaption, style, null);
        HANDLE parent;
        int vleft = std.c.windows.windows.CW_USEDEFAULT;
        int vtop = std.c.windows.windows.CW_USEDEFAULT;
        int vwidth = width ();
        int vheight = height ();

        if (this.parent () !== null)
            parent = this.parent ().digPlatformHWND;
        if (digPlatformFullscreen)
        {
            parent = (HANDLE) 0;
            vleft = 0;
            vtop = 0;
        }

        digPlatformHWNDCreate (WS_EX_APPWINDOW, digPlatformClassName, digPlatformCaption, digPlatformStyle, vleft, vtop, vwidth, vheight, parent, (_HANDLE) 0, hinst, null);
        digPlatformCreateContext ();
        digPlatformMoved ();
        childIterate (delegate void (Control child) { child.digPlatformRecreate (); });
        makeCurrent ();
    }+/

/+
#endif
+/
}

extern (Windows)
{
    HANDLE wglCreateContext (HDC hdc);
    BOOL wglMakeCurrent (HDC hdc, HGLRC hglrc);
    BOOL wglDeleteContext (HGLRC hglrc);
    void *wglGetProcAddress (LPCSTR lpszProc);
    HANDLE wglGetCurrentContext ();
    HDC wglGetCurrentDC ();
}

extern (Windows)
{
    void glAlphaFunc (GLenum func, GLclampf ref);
    void glBegin (GLenum mode);
    void glBindTexture (GLenum target, GLuint texture);
    void glBlendFunc (GLenum sfactor, GLenum dfactor);
    void glCallList (GLuint list);
    void glClear (GLbitfield mask);
    void glClearColor (GLclampf red, GLclampf green, GLclampf blue, GLclampf alpha);
    void glClearDepth (GLclampd depth);
    void glColor3f (GLfloat red, GLfloat green, GLfloat blue);
    void glColor4f (GLfloat red, GLfloat green, GLfloat blue, GLfloat alpha);
    void glColor4ub (GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha);
    void glColorMask (GLboolean red, GLboolean green, GLboolean blue, GLboolean alpha);
    void glColorPointer (GLint size, GLenum type, GLsizei stride, GLvoid *pointer);
    void glCopyTexImage2D (GLenum target, GLint level, GLenum internalFormat, GLint x, GLint y, GLsizei width, GLsizei height, GLint border);
    void glCullFace (GLenum mode);
    void glDeleteLists (GLuint list, GLsizei range);
    void glDeleteTextures (GLsizei n, GLuint *textures);
    void glDepthFunc (GLenum func);
    void glDepthMask (GLboolean value);
    void glDepthRange (GLclampd zNear, GLclampd zFar);
    void glDisable (GLenum cap);
    void glDisableClientState (GLenum cap);
    void glDrawArrays (GLenum mode, GLint first, GLsizei count);
    void glDrawElements (GLenum mode, GLsizei count, GLenum type, GLvoid *indices);
    void glEnable (GLenum cap);
    void glEnableClientState (GLenum cap);
    void glEnd ();
    void glEndList ();
    void glFogf (GLenum pname, GLfloat param);
    void glFogfv (GLenum pname, GLfloat *params);
    void glFogi (GLenum pname, GLint param);
    void glFogiv (GLenum pname, GLint *params);
    void glFlush ();
    void glFrontFace (GLenum mode); 
    GLuint glGenLists (GLsizei range);
    void glGenTextures (GLsizei n, GLuint *textures);
    void glGetBooleanv (GLenum pname, GLboolean *params);
    void glGetDoublev (GLenum pname, GLdouble *params);
    GLenum glGetError ();
    void glGetFloatv (GLenum pname, GLfloat *params);
    void glGetIntegerv (GLenum pname, GLint *params);
    void glGetLightfv (GLenum light, GLenum pname, GLfloat *params);
    void glGetLightiv (GLenum light, GLenum pname, GLint *params);
    void glGetMaterialfv (GLenum face, GLenum pname, GLfloat *params);
    void glGetMaterialiv (GLenum face, GLenum pname, GLint *params);
    GLubyte *glGetString (GLenum name);
    void glGetTexEnvfv (GLenum target, GLenum pname, GLfloat *params);
    void glGetTexEnviv (GLenum target, GLenum pname, GLint *params);
    void glGetTexImage (GLenum target, GLint level, GLenum format, GLenum type, GLvoid *pixels);
    void glGetTexLevelParameterfv (GLenum target, GLint level, GLenum pname, GLfloat *params);
    void glGetTexLevelParameteriv (GLenum target, GLint level, GLenum pname, GLint *params);
    void glGetTexParameterfv (GLenum target, GLenum pname, GLfloat *params);
    void glGetTexParameteriv (GLenum target, GLenum pname, GLint *params);
    void glHint (GLenum target, GLenum mode);
    GLboolean glIsEnabled (GLenum cap);
    void glLightfv (GLenum light, GLenum pname, GLfloat *params);
    void glLightModelf (GLenum pname, GLfloat param);
    void glLightModelfv (GLenum pname, GLfloat *param);
    void glLightModeli (GLenum pname, GLint param);
    void glLineWidth (GLfloat value);
    void glLoadIdentity ();
    void glLogicOp (GLenum opcode);
    void glMaterialfv (GLenum face, GLenum pname, GLfloat *params);
    void glMatrixMode (GLenum mode);
    void glMultMatrixd (GLdouble *m);
    void glMultMatrixf (GLfloat *m);
    void glNewList (GLuint list, GLenum mode);
    void glNormal3f (GLfloat x, GLfloat y, GLfloat z);
    void glNormal3fv (GLfloat *v);
    void glNormalPointer (GLenum type, GLsizei stride, GLvoid *pointer);
    void glPixelStoref (GLenum pname, GLfloat param);
    void glPixelStorei (GLenum pname, GLint param);
    void glPointSize (GLfloat size);
    void glPolygonMode (GLenum face, GLenum mode);
    void glPolygonOffset (GLfloat factor, GLfloat units);
    void glPopMatrix ();
    void glPushMatrix ();
    void glReadPixels (GLint x, GLint y, GLsizei width, GLsizei height, GLenum format, GLenum type, GLvoid *pixels);
    void glRotatef (GLfloat angle, GLfloat x, GLfloat y, GLfloat z);
    void glScalef (GLfloat x, GLfloat y, GLfloat z);
    void glStencilFunc (GLenum func, GLint ref, GLuint mask);
    void glStencilOp (GLenum fail, GLenum zfail, GLenum zpass);
    void glTexCoord2f (GLfloat x, GLfloat y);
    void glTexCoordPointer (GLint size, GLenum type, GLsizei stride, GLvoid *pointer);
    void glTexEnvf (GLenum target, GLenum pname, GLfloat param);
    void glTexEnvfv (GLenum target, GLenum pname, GLfloat *params);
    void glTexEnvi (GLenum target, GLenum pname, GLint	param);
    void glTexEnviv (GLenum target, GLenum pname, GLint *params);
    void glTexGend (GLenum coord, GLenum pname, GLdouble param);
    void glTexGendv (GLenum coord, GLenum pname, GLdouble *params);
    void glTexGenf (GLenum coord, GLenum pname, GLfloat param);
    void glTexGenfv (GLenum coord, GLenum pname, GLfloat *params);
    void glTexGeni (GLenum coord, GLenum pname, GLint param);
    void glTexGeniv (GLenum coord, GLenum pname, GLint *params);
    void glTexImage1D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLint border, GLenum format, GLenum type, GLvoid *pixels);
    void glTexImage2D (GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, GLvoid *pixels);
    void glTexParameterf (GLenum target, GLenum pname, GLfloat param);
    void glTexParameteri (GLenum target, GLenum pname, GLint param);
    void glTranslatef (GLfloat x, GLfloat y, GLfloat z);
    void glVertex2f (GLfloat x, GLfloat y);
    void glVertex3f (GLfloat x, GLfloat y, GLfloat z);
    void glVertex4f (GLfloat x, GLfloat y, GLfloat z, GLfloat w);
    void glVertexPointer (GLint size, GLenum type, GLsizei stride, GLvoid *pointer);
    void glViewport (GLint x, GLint y, GLsizei width, GLsizei h);
}
